import { NextResponse } from 'next/server'
import { db } from '@/lib/db'

export async function POST(request: Request) {
  try {
    const body = await request.json()
    const {
      gameId,
      pricingPlanId,
      amount,
      currency,
      customerName,
      customerEmail,
      customerPhone,
      notes,
      paymentMethod,
    } = body

    // Validate required fields
    if (!gameId || !pricingPlanId || !amount || !customerName || !customerEmail || !customerPhone || !paymentMethod) {
      return NextResponse.json(
        { error: 'Missing required fields' },
        { status: 400 }
      )
    }

    // Create the order
    const order = await db.order.create({
      data: {
        gameId,
        pricingPlanId,
        amount: parseFloat(amount),
        currency: currency || 'IDR',
        customerName,
        customerEmail,
        customerPhone,
        notes,
        paymentMethod,
        status: 'PENDING',
      },
      include: {
        game: true,
        pricingPlan: true,
      },
    })

    return NextResponse.json(order)
  } catch (error) {
    console.error('Failed to create order:', error)
    return NextResponse.json(
      { error: 'Failed to create order' },
      { status: 500 }
    )
  }
}

export async function GET() {
  try {
    const orders = await db.order.findMany({
      include: {
        game: true,
        pricingPlan: true,
      },
      orderBy: {
        createdAt: 'desc',
      },
    })

    return NextResponse.json(orders)
  } catch (error) {
    console.error('Failed to fetch orders:', error)
    return NextResponse.json(
      { error: 'Failed to fetch orders' },
      { status: 500 }
    )
  }
}